unit IWGrids32;
{PUBDIST}

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing,
  {$ENDIF}

  Classes,
  {$IFDEF VSNET}
  {$ELSE}
  {$IFDEF Linux}QControls, {$ELSE}Controls, {$ENDIF}
  {$ENDIF}
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  IWGridCommon, IWControl32, IWFont, IWTypes, IWHTMLTag, IWRenderContext,
  IWBaseInterfaces;

type
  TIWGridCell32 = class;
  TIWCustomGrid32 = class;

  // This is a TCollectionItem so in the future it can be made visual easier later.
  TIWGridCell32 = class(TCollectionItem)
  //COLSPAN=Number (columns spanned by the cell)
  //ROWSPAN=Number (rows spanned by the cell)\
  private
    FGrid: TIWCustomGrid32;
  protected
    FAlignment: TAlignment;
    FControl: TIWCustomControl32;
    FFont: TIWFont;
    FHeader: Boolean;
    FHeight: string;
    FText: string;
    FTag: TObject;
    FVAlign: TIWGridVAlign;
    FVisible: Boolean;
    FWidth: string;
    FWrap: Boolean;
    FRawText : Boolean;
    //
    procedure AssignTo(ADest: TPersistent); override;
    procedure SetGrid(const AValue: TIWCustomGrid32);
    procedure SetControl(const AValue: TIWCustomControl32);
    procedure SetFont(const AValue: TIWFont);
    procedure SetHeight(const AValue: string);
    procedure SetWidth(const AValue: string);
  public
    constructor Create(ACollection: TCollection); override;
    destructor Destroy; override;
    function RenderSelf(AGrid: TIWCustomGrid32; const ARow: Integer;
      const AColumn: Integer; AContext: TIWComponent32Context; AText: string = ''): TIWHTMLTag; virtual;

    property Grid: TIWCustomGrid32 read FGrid write SetGrid;
  published
    property Alignment: TAlignment read FAlignment write FAlignment;
    property Control: TIWCustomControl32 read FControl write SetControl;
    property Font: TIWFont read FFont write SetFont;
    property Header: Boolean read FHeader write FHeader;
    property Height: string read FHeight write SetHeight;
    property Tag: TObject read FTag write FTag;
    property Text: string read FText write FText;
    property VAlign: TIWGridVAlign read FVAlign write FVAlign;
    property Visible: Boolean read FVisible write FVisible;
    property Width: string read FWidth write SetWidth;
    property Wrap: Boolean read FWrap write FWrap;
    property RawText : Boolean read FRawText write FRawText;
  end;

  TIWGridCells32 = class(TOwnedCollection)
  public

    function Add: TIWGridCell32;
    // All this is for compatibility with Delphi 5 becouese Owner property is not defined in
    // TOWnedCollection

    function Owner: TPersistent;
  end;

  TIWOnRenderCell32 = procedure(ACell: TIWGridCell32; const ARow: Integer; const AColumn: Integer)
    of object;

  TIWCustomGrid32 = class(TIWCustomControl32)
  protected
    FBorderSize: Integer;
    FCaption: string;
    FCells: array of array of TIWGridCell32;
    FCellPadding: Integer;
    FCellSpacing: Integer;
    FFrameBuffer: Integer;
    FLines: TIWGridLines;
    FOnRenderCell: TIWOnRenderCell32;
    FUseSize: Boolean;
    //
    procedure DoRenderCell(ACell: TIWGridCell32; const ARow: Integer; const AColumn: Integer);
    procedure RenderCells(AContext: TIWComponent32Context; AGridTag: TIWHTMLTag); virtual; abstract;
    function IsValidCell(ACell: TIWGridCell32): Boolean; virtual; abstract;
    procedure InitControl; override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property BorderSize: Integer read FBorderSize write FBorderSize;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property Caption: string read FCaption write FCaption;
    property CellPadding: Integer read FCellPadding write FCellPadding;
    property CellSpacing: Integer read FCellSpacing write FCellSpacing;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    property FrameBuffer: Integer read FFrameBuffer write FFrameBuffer;
    property Lines: TIWGridLines read FLines write FLines;
    property OnRenderCell: TIWOnRenderCell32 read FOnRenderCell write FOnRenderCell;
    property UseSize: Boolean read FUseSize write FUseSize;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWGrid32.bmp}
  TIWGrid32 = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWGrid32), 'TIWGrid32.bmp')]
  {$ENDIF}
  TIWGrid32 = class(TIWCustomGrid32, IIWSubmitControl)
  protected
    FSubmitParam : String;
    FCellCollection: TIWGridCells32;
    FColumnCount: Integer;
    FRowCount: Integer;
    //
    function GetCell(const ARow, AColumn: Integer): TIWGridCell32;
    procedure RenderCells(AContext: TIWComponent32Context; AGridTag: TIWHTMLTag); override;
    procedure SetColumnCount(const AValue: Integer);
    procedure SetRowCount(const AValue: Integer);
    procedure Submit(const AValue: string); override;
    function IsValidCell(ACell: TIWGridCell32): Boolean; override;
    procedure InitControl; override;
    procedure InitDesignTime; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    // procedure ClearRendering; override;

    function CellExists(const ARow, AColumn: Integer): Boolean;
    function GetSubmitParam : String;
    procedure DeleteColumn(const AColumn: Integer);
    procedure DeleteRow(const ARow: Integer);

    property Cell[const ARow: Integer; const AColumn: Integer]: TIWGridCell32 read GetCell;
  published
    property ColumnCount: Integer read FColumnCount write SetColumnCount;
    property RowCount: Integer read FRowCount write SetRowCount;
  end;


implementation

uses
  SysUtils,
  IWForm32, IWResourceStrings, IWHTML32Interfaces, SWSystem, SWStrings,
  IWBaseHTMLControl;

{ TIWGridCell32 }

procedure TIWGridCell32.AssignTo(ADest: TPersistent);
begin
  if ADest is TIWGridCell32 then begin
    with TIWGridCell32(ADest) do begin
      Alignment := Self.Alignment;
      Font.Assign(Self.Font);
      Header := Self.Header;
      Height := Self.Height;
      Text := Self.Text;
      VAlign := Self.VAlign;
      Width := Self.Width;
      Wrap := Self.Wrap;
      Visible := Self.Visible;
      Control := Self.Control;
    end;
  end else begin
    inherited;
  end;
end;

constructor TIWGridCell32.Create(ACollection: TCollection);
begin
  inherited Create(ACollection);
  FFont := TIWFont.Create;
  // Copy from Grids font.
  FFont.Assign(Font);
  FHeight := '0';
  FVisible := True;
  FWidth := '0';
  FControl := nil;
  FRawText := True;
end;

destructor TIWGridCell32.Destroy;
begin
  FreeAndNil(FFont);
  inherited Destroy;
end;

function TIWGridCell32.RenderSelf(AGrid: TIWCustomGrid32; const ARow,
  AColumn: Integer; AContext: TIWComponent32Context; AText: string): TIWHTMLTag;
var
  LControlRenderData: TIWHTMLTag;
  LVAlign: string;
  LControlContext: TIWComponent32Context;
begin
  if not Visible then begin
    Result := TIWHTMLTag.CreateTag('TD'); try
      Result.Contents.AddText('&nbsp;');
    except FreeAndNil(Result); raise; end;
  end else begin
    FGrid := AGrid; try
      AGrid.DoRenderCell(Self, ARow, AColumn);
      LControlRenderData := nil;
      if Control <> nil then begin
        LControlContext := TIWComponent32Context.Create(Control, AContext.ContainerContext, AContext.PageContext);
        LControlRenderData := Control.RenderHTML(LControlContext);
        LControlRenderData.AddParmsList(Control.ExtraTagParams);
        if Assigned(LControlRenderData) then begin
          HTML32ControlInterface(Control).DoHTMLTag(LControlRenderData);
//          LTmp := Control.RenderStyle;
//          if LControlRenderData.Params.Values['STYLE'] <> '' then begin
//            LTmp := LTmp + ' ' + LControlRenderData.Params.Values['STYLE'] + ';';
//          end;
//          if LControlRenderData.Params.Values['ID'] = '' then begin
//            LControlRenderData.AddStringParam('ID', Control.HTMLName);
//          end;
          if LControlRenderData.Params.Values['NAME'] = '' then begin
            LControlRenderData.AddStringParam('NAME', Control.HTMLName);
          end;
//          if Control.Visible and (LControlRenderData.Params.Values['CLASS'] = '') then begin
//            LControlRenderData.AddStringParam('CLASS', Control.RenderCSSClass);
//          end;
//          if not Control.Visible or not AnsiSameText(Control.RenderCSSClass, Control.HTMLName + 'CSS') then begin
//            LControlRenderData.AddStringParam('STYLE', LTmp);
//          end;
//          Control.RenderScripts(LControlRenderData, false);
        end;
        AContext.ContainerContext.AddComponent(LControlContext, nil);
      end;
      // VAlign
      LVAlign := '';
      case FVAlign of
        vaMiddle: LVAlign := 'VAlign=middle';
        vaTop: LVAlign := 'VAlign=top';
        vaBottom: LVAlign := 'VAlign=bottom';
        vaBaseline: LVAlign := 'VAlign=baseline';
      end;
      //
      if Length(AText) = 0 then begin
        AText := Text;
      end;

      if not RawText then begin
        AText := TIWBaseHTMLControl.TextToHTML(AText, true, false);
      end;

      AText := iif((Trim(AText) = '') and (LControlRenderData = nil), '&nbsp;', AText);

      Result := TIWHTMLTag.CreateTag(iif(Header, 'TH', 'TD')); try
        Result.Add(LVAlign);
        // Always render alignment, default is different or TH and TD
        Result.AddStringParam('align', IWGridAlignments[Alignment]);
        Result.Add(iif(not Wrap, 'NOWRAP'));
        Result.Add(iif((Width <> '') and (Width <> '0'), 'Width=' + Width));
        Result.Add(iif((Height <> '') and (Height <> '0'), 'Height=' + Height));
//        Result.Add(TIWControl.HTMLColor(BGColor, 'BGColor'));
//        Result.Add(iif(Length(Font.CSSStyle) > 0, 'CLASS=' + Font.CSSStyle));

        // FontToString checks Font.Enabled
{        if Font.CSSStyle = '' then
        begin
           if Result.Contents.AddTagAsObject(Font.FontToStringTag(AText)) = nil then
              Result.Contents.AddText(AText);
        end
        else
        begin
           Result.Contents.AddText(AText);
        end;}
        if Trim(AText) <> '' then begin
          Result.Contents.AddTagAsObject(Font.FontToStringTag32(AText));
        end;

        if Assigned(LControlRenderData) then begin
          Result.Contents.AddTagAsObject(LControlRenderData);
        end;
      except FreeAndNil(Result); raise; end;
    finally FGrid := nil; end;
  end;
end;

procedure TIWGridCell32.SetControl(const AValue: TIWCustomControl32);
begin
  if AValue <> nil then begin
{$IFNDEF VSNET}
    if not (csDesigning in AValue.ComponentState) then begin
      AValue.Visible := False;
      AValue.Parent := nil;
    end;
{$ENDIF}
    if AValue is TIWCustomGrid32 then begin
      if FGrid = AValue then begin
        raise Exception.Create(RSCanSetControlToSelf);
      end;
      with TIWCustomGrid32(AValue) do begin
        if IsValidCell(self) then begin
          raise Exception.Create(RSCanSetControlToSelf);
        end;
      end;
    end;
  end;
  FControl := AValue;
end;

procedure TIWGridCell32.SetFont(const AValue: TIWFont);
begin
  FFont.Assign(AValue);
end;

procedure TIWGridCell32.SetGrid(const AValue: TIWCustomGrid32);
begin
  FGrid := AValue;
end;

procedure TIWGridCell32.SetHeight(const AValue: string);
var
  s: string;

  procedure InvalidHeight;
  begin
    raise Exception.Create(Format(RSInvalidHeight, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidHeight;
  // Percentage must be >0
  end else if (s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0) then begin
    FHeight := AValue;
  end else if StrToIntDef(s, -1) >= 0 then begin
    FHeight := AValue;
  end else begin
    InvalidHeight;
  end;
end;

procedure TIWGridCell32.SetWidth(const AValue: string);
var
  s: string;

  procedure InvalidWidth;
  begin
    raise Exception.Create(Format(RSInvalidWidth, [s]));
  end;

begin
  s := Trim(AValue);
  if Length(s) = 0 then begin
    InvalidWidth;
  // Percentage must be >0
  end else if (s[Length(s)] = '%') and (StrToIntDef(Copy(s, 1, Length(s) - 1), 0) > 0) then begin
    FWidth := AValue;
  end else if StrToIntDef(s, -1) >= 0 then begin
    FWidth := AValue;
  end else begin
    InvalidWidth;
  end;
end;

{ TIWCustomGrid32 }

procedure TIWCustomGrid32.InitControl;
begin
  inherited;
  FBorderSize := 1;
  FFrameBuffer := 40;
  FNeedsFormTag := True;
  FUseSize := True;

  FCaption := '';

  Width := 300;
  Height := 150;
end;

procedure TIWCustomGrid32.DoRenderCell(ACell: TIWGridCell32; const ARow,
  AColumn: Integer);
begin
  if Assigned(OnRenderCell) then begin
    OnRenderCell(ACell, ARow, AColumn);
  end;
end;

function TIWCustomGrid32.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  // If Control property is used.
  Result := TIWHTMLTag.CreateTag('TABLE');
  try
    if BorderSize >= 0 then begin
      Result.AddIntegerParam('Border', BorderSize);
    end;
 //   Result.AddColor('BGColor', BGColor);
    Result.AddIntegerParam('CELLPADDING', CellPadding);
    Result.AddIntegerParam('CELLSPACING', CellSpacing);
    if UseSize then begin
      Result.AddIntegerParam('WIDTH', Width);
    end;
    //TODO: Convert ot 3.2
  {  if BorderSize > 0 then begin
      Result.AddColor('BORDERCOLOR', BorderColors.Color);
      Result.AddColor('BORDERCOLORLIGHT', BorderColors.Light);
      Result.AddColor('BORDERCOLORDARK', BorderColors.Dark);
      case Lines of
        tlNone: Result.AddStringParam('RULES', 'none');
        tlRows: Result.AddStringParam('RULES', 'rows');
        tlCols: Result.AddStringParam('RULES', 'cols');
      end;
    // until here.
    end;
}
     {Result.Contents.AddText(iif(Caption, '<CAPTION>' + Caption + '</CAPTION>'));}
     if Caption <> '' then
     begin
        Result.Contents.AddText('<CAPTION>');
        Result.Contents.AddTagAsObject(WebFont.FontToStringTag32(Caption));
        Result.Contents.AddText('</CAPTION>');
     end;
    RenderCells(TIWComponent32Context(AContext), Result);
  except FreeAndNil(Result); raise; end;
end;

{ TIWGridCells32 }

function TIWGridCells32.Add: TIWGridCell32;
begin
  Result := TIWGridCell32(inherited Add);
end;

function TIWGridCells32.Owner: TPersistent;
begin
  Result := inherited GetOwner;
end;

{ TIWGrid32 }

function TIWGrid32.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

function TIWGrid32.CellExists(const ARow, AColumn: Integer): Boolean;
begin
  if (AColumn >= ColumnCount) or (ARow >= RowCount) or (AColumn < 0) or (ARow < 0) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  Result := FCells[ARow, AColumn] <> nil;
end;

{procedure TIWGrid32.ClearRendering;
Var
  i, j: Integer;
begin
  inherited ClearRendering;
  for i := Low(FCells) to High(FCells) do begin
    for j := Low(FCells[i]) to High(FCells[i]) do begin
      if CellExists(i, j) then begin
        if Assigned(FCells[i, j].Control) then begin
          FCells[i, j].Control.ClearRendering;
        end;
      end;
    end;
  end;
end;}

procedure TIWGrid32.InitDesignTime;
begin
  inherited;
  if not IsDesignMode then begin
    SetLength(FCells, 1);
    SetLength(FCells[0], 1);
  end;
end;

procedure TIWGrid32.InitControl;
begin
  inherited;
  FCellCollection := TIWGridCells32.Create(Self, TIWGridCell32);
  // Bypass setter routines
  FColumnCount := 1;
  FRowCount := 1;
end;

procedure TIWGrid32.DeleteColumn(const AColumn: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AColumn < 0) or (AColumn >= ColumnCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;

  for i := 0 to High(FCells) do begin
    for j := AColumn to High(FCells[i]) - 1 do begin
      if (j = AColumn) and CellExists(i, j) then begin
        FCells[i, j].Free;
      end;
      FCells[i, j] := FCells[i, j + 1];
    end;
  end;

  for i := 0 to High(FCells) do begin
    FCells[i, ColumnCount - 1] := nil;
  end;
  // This will call SetColumnCount
  ColumnCount := ColumnCount - 1;
end;

procedure TIWGrid32.DeleteRow(const ARow: Integer);
var
  i, j: Integer;
begin
  if (ARow < 0) or (ARow >= RowCount) then begin
    raise Exception.Create(RSIndexOutOfBounds);
  end;
  for i := High(FCells[ARow]) downto 0 do begin
    if CellExists(ARow, i) then begin
      FCellCollection.Delete(FCells[ARow, i].Index);
    end;
  end;
  for i := ARow to High(FCells) - 1 do begin
    for j := 0 to High(FCells[i]) do begin
      FCells[i, j] := FCells[i + 1, j];
    end;
  end;
  for i := 0 to High(FCells[RowCount - 1]) do begin
    FCells[RowCount - 1, i] := nil;
  end;
  // This will call SetRowCount
  RowCount := RowCount - 1;
end;

procedure TIWGrid32.Dispose(ADispose: Boolean); 
begin
  FreeAndNil(FCellCollection);
  inherited;
end;

function TIWGrid32.GetCell(const ARow, AColumn: Integer): TIWGridCell32;
begin
  if not CellExists(ARow, AColumn) then begin
    FCells[ARow, AColumn] := FCellCollection.Add;
    // this will copy current Grid font settings to all cells after they were created
    // Later in OnRenderCell the user program can change these settings by loading different values in
    // GridCell.Font
    FCells[ARow, AColumn].Font.Assign(Self.WebFont);
  end;
  Result := FCells[ARow, AColumn];
end;

function TIWGrid32.IsValidCell(ACell: TIWGridCell32): Boolean;
Var
  LCell: TCollectionItem;
begin
  LCell := FCellCollection.FindItemID(ACell.ID);
  result := LCell = ACell;
end;

procedure TIWGrid32.RenderCells(AContext: TIWComponent32Context; AGridTag: TIWHTMLTag);
var
  i: Integer;
  j: Integer;
begin
  for i := 0 to RowCount - 1 do begin
    with AGridTag.Contents.AddTag('TR') do begin
      for j := 0 to ColumnCount - 1 do begin
        if CellExists(i, j) then begin
          with FCells[i, j] do begin
            if Visible then begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
            end else begin
                Contents.AddTagAsObject(RenderSelf(Self, i, j, AContext));
            end;
          end;
        end else begin
          // Not &nbsp; - only if cell exists but is empty
          with Contents.AddTag('TD') do
            Contents.AddText('');
        end;
      end;
    end;
  end;
end;

procedure TIWGrid32.SetColumnCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AValue <> FColumnCount) and (AValue > 0) then begin
    if not IsDesignMode then begin
      for i := 0 to High(FCells) do begin
        if AValue < FColumnCount then begin
          for j := ColumnCount - 1 downto AValue do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i,j].Index);
            end;
          end;
        end;
        SetLength(FCells[i], AValue);
        if AValue > FColumnCount then begin
          for j := FColumnCount to AValue - 1 do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    end;
    FColumnCount := AValue;
  end;
end;

procedure TIWGrid32.SetRowCount(const AValue: Integer);
var
  i: Integer;
  j: Integer;
begin
  if (AValue <> FRowCount) and (AValue > 0) then begin
    if not IsDesignMode then begin
      if AValue < FRowCount then begin
        for i := FRowCount - 1 downto AValue do begin
          for j := ColumnCount - 1 downto 0 do begin
            if Assigned(FCells[i, j]) then begin
              FCellCollection.Delete(FCells[i, j].Index);
            end;
          end;
        end;
      end;
      SetLength(FCells, AValue);
      if AValue > FRowCount then begin
        for i := FRowCount to AValue - 1 do begin
          SetLength(FCells[i], FColumnCount);
          for j := 0 to High(FCells[i]) do begin
            FCells[i, j] := nil;
          end;
        end;
      end;
    end;
    FRowCount := AValue;
  end;
end;

procedure TIWGrid32.Submit(const AValue: string);
var
  s: string;
  {LColumn: Integer;
  LRow: Integer;}
begin
  FSubmitParam := AValue;
  s := AValue;
  {LRow := StrToInt(Fetch(s, '_'));
  LColumn := StrToInt(s);}
end;

end.
